/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var logger = require( 'debug' );
var isNonNegativeInteger = require( '@stdlib/assert/is-nonnegative-integer' ).isPrimitive;
var isNonNegativeIntegerArray = require( '@stdlib/assert/is-nonnegative-integer-array' ).primitives;


// VARIABLES //

var debug = logger( 'plot:set:symbols-size' );


// MAIN //

/**
* Sets the symbols size.
*
* @private
* @param {(NonNegativeInteger|Array<NonNegativeInteger>)} v - size
* @throws {TypeError} must be a nonnegative integer or nonnegative integer array
* @returns {void}
*/
function set( v ) {
	/* eslint-disable no-invalid-this */
	var isInt = isNonNegativeInteger( v );
	if ( !isInt && !isNonNegativeIntegerArray( v ) ) {
		throw new TypeError( 'invalid value. `symbolsSize` must be a nonnegative integer or nonnegative integer array. Value: `' + v + '.`' );
	}
	if ( isInt ) {
		v = [ v ];
	} else {
		v = v.slice();
	}
	debug( 'Current value: %s.', JSON.stringify( this._symbolsSize ) );

	this._symbolsSize = v;
	debug( 'New Value: %s.', JSON.stringify( this._symbolsSize ) );

	this.emit( 'change' );
}


// EXPORTS //

module.exports = set;
